## -*-Tcl-*-
 # ###################################################################
 #  AlphaTk - the ultimate editor
 # 
 #  FILE: "alpha_commands.tcl"
 #                                    created: 04/12/98 {23:17:46 PM} 
 #                                last update: 1999-09-05T19:31:50Z 
 #  Author: Vince Darley
 #  E-mail: vince@santafe.edu
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: http://www.santafe.edu/~vince
 #  
 # Copyright (c) 1998-1999  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 #  Description: 
 # 
 #  History
 # 
 #  modified by  rev reason
 #  -------- --- --- -----------
 #  04/12/98 VMD 1.0 original
 # ###################################################################
 ##
#==============================================================================
#= Alpha Commands
#==============================================================================
#
#In this list of routines, text between '<' and '>' is a placeholder for a 
#required parameter, text between '[' and ']' is a placeholder for an 
#optional parameter, and the '|' signifies a choice of two or more 
#alternatives.  A '+' signifies that the previous symbol can be present one 
#or more times, while a '*" means zero or more times.  Some commands have no 
#parameters, and are only expected to be called interactively.
#
#
#
#The following are Alpha-specific tcl routines:
#

#  macros  #

# dumpMacro - prompts for a name and then dumps a tcl proc representation 
#  of the current keyboard macro into the current window.
proc dumpMacro {args} {echo "dumpMacro $args"}
# endKeyboardMacro - stop recording keyboard macro
proc endKeyboardMacro {args} {echo "endKeyboardMacro $args"}
# executeKeyboardMacro - execute the current keyboard 
#  macro
proc executeKeyboardMacro {args} {echo "executeKeyboardMacro $args"}
# startKeyboardMacro - start recording keyboard macro
proc startKeyboardMacro {args} {echo "startKeyboardMacro $args"}


#  other  #

proc addAlphaChars {args} {echo "addAlphaChars $args"}

# colors <fore red> <fore green> <fore blue> <back red> <back green> <back blue>
proc colors {args} {echo "colors $args"}
# displayMode <mode> - Up to four characters of the 'mode' string are 
#  displayed in the status line at the bottom of a window.
proc displayMode {m} {
    global mode alpha::adjust_mode
    set alpha::adjust_mode $mode
    if {$m != ""} {
	global tcl_platform
	switch -- $tcl_platform(platform) {
	    "windows" -
	    "unix" {
		place .status.mode -in .status -x [expr {[winfo screenwidth .status] -100}] -y 0 -height 21 -width 60
	    }
	    "macintosh" {
		place .status.mode -in .status -x [expr {[winfo screenwidth .status] -100}] -y -4 -height 21 -width 60
	    }
	}
	.status.mode.menu invoke $m
    } else {
	place forget .status.mode
    }    
    #.status.mode configure -text $m
}

# getFileInfo <file> <arr> - Given a file name, creates an array called 
#  'arr' in the current context, containing fields 'created', 'creator', 
#  'modified', 'type', 'datalen', and 'resourcelen'. 'created' and 
#  'modified' are in a form suitable for the command 'mtime'.
proc getFileInfo {f a} {
    upvar $a arr
    file stat $f arr
    regsub -all -- "-" [file attributes $f] "" l
    array set arr $l
    set arr(creator) $f
    set arr(type) TEXT
    set arr(modified) [file mtime $f]
    set arr(datalen) [file size $f]
    set arr(created) $arr(ctime)
}

# setFileInfo <file> <field> [arg] - Allows some information to be set 
#  for files. Settable fields are 'modified', 'created', 'creator', and 
#  'type' all take an argument. 'resourcelen' can be set, but doesn't take 
#  an argument and just removes the resource fork.
proc setFileInfo {file f {arg ""}} {
    if {![file exists $file]} {
	error "No such file: $file"
    }
    switch -- $f {
	"modified" {
	    echo "Unimplemented: setFileInfo $file modified"
	}
	"created" {
	    echo "Unimplemented: setFileInfo $file created"
	}
	"resourcelen" {
	    echo "Unimplemented: setFileInfo $file resourcelen"
	}
	default {
	    file attributes $file -$f $arg
	}
    }
}

proc getModifiers {args} {return 0}
# insertColorEscape <pos> <color ind> [hypertext func] - Create a color 
#  or style "point" for documentation purposes. Look at the file 
#  "docColors.tcl" for examples. The hypertext func is only used when the 
#  "point" is underline. See 'getColors' for info about the current file.
proc insertColorEscape {pos color {func ""}} {
    global alphaPriv
    if {$color == 0 || $color == 12} {
	text::color \
	  $alphaPriv(insertColorEscapePos) $pos \
	  color$alphaPriv(insertColorEscapeColor) \
	  $alphaPriv(insertColorEscapeFunc)
	unset alphaPriv(insertColorEscapePos) 
	unset alphaPriv(insertColorEscapeColor)
	unset alphaPriv(insertColorEscapeFunc) 
    } else {
	set alphaPriv(insertColorEscapePos) $pos
	set alphaPriv(insertColorEscapeColor) $color
	set alphaPriv(insertColorEscapeFunc) $func
    }
}
namespace eval text {}

proc text::hyper {from to hyper} {
    text::color $from $to 15 $hyper
}

proc text::color {from to colour {hyper ""}} {
    text_cmd tag add color$colour $from $to
    if {$colour == 15} {
	# do hyper
	text_cmd addHyper $from $to $hyper
    }
}

# This isn't required by Tcl only Alpha.
proc linkVar {args} {}

proc beep {args} {bell}

# evaluate - loads hilited text, or entire window if 
#  nothing is hilited. 'load'ing means that whatever 
#  bindings or macro definitions are present in the 
#  loaded text take effect
proc evaluate {} {
    global win::tk win::Current
    if {[llength [set sel [selectLimits]]] == 0} {
	set sel [list [minPos] [maxPos]]
    }
    if {[catch {uplevel \#0 [eval [list tw::get $win::tk($win::Current)] \
      $sel]} err]} {
	message "Error: $err"
    } else {
	message "Result: $err"
    }
}

proc selectLimits {} {
    if {[set res [text_cmd tag ranges sel]] == ""} {
	return [text_cmd tag ranges backsel]
    } else {
	return $res
    }
}

# regModeKeywords [options] <mode> <keyword list> - Set keywords and comments 
#  that 
#  Alpha can recognize to color them. Specifically, in mode <mode>, every 
#  keyword specified in the list is colored non-black (blue, by default). 
#  Comments, if specified by '-e' or '-b' below, are colored red by defualt.
#    -a				Keywords will be *added* to existing mode
#    				keywords. The new keywords can be a different
#    				color than older keywords. This flag can also be
#    				used to modify other attributes, but it cannot be
#    				used to modify colors of existing keywords.
#	-m <c>			Specify a magic character. Every word beginning with the 
#				magic character is a keyword.
#	-e <commentstr>		Specify a string that begins comments that last to 
#				the end of the line. 
#	-b <comment beg> <comment end>	Specify a pair of strings that bracket a 
#					comment.
#	-c <color>		Comment color.
#	-k <color>		Keyword color.
#	-s <color>		String color. Alpha can color single-line
#				strings only, using the simple heuristic
#				of assuming the first two double quotes
#				constitute a string. 
#	-i <char>		Specify a character to display differently.
#				Commonly used for curly braces, etc.
#	-I <color>		Color of above characters.
#	Keywords must be less than 20 characters long.
proc regModeKeywords {args} {
    getOpts {-m -e {-b 2} -c -f -k -s -i -I} "lappend"
    set mode [lindex $args 0]
    global ${mode}::keywords ${mode}::keywordsopts
    if {![info exists opts(-a)]} {
	if {[info exists ${mode}::keywords]} {
	    unset ${mode}::keywords
	}
    } else {
	# new options override all old options which we stored in
	# the given array
	if {[info exists ${mode}::keywordsopts]} {
	    foreach o [array names ${mode}::keywordsopts] {
		if {![info exists opts($o)]} {
		    set opts($o) [set ${mode}::keywordsopts($o)]
		}
	    }
	}
    }
    if {![info exists opts(-k)]} {
	set opts(-k) blue
    }
    set col [lsearch -exact {blue cyan green magenta red white yellow} $opts(-k)]
    incr col
    if {$col == 0} {set col 1}
    
    foreach kw [lindex $args 1] {
	set ${mode}::keywords($kw) color$col
    }
    if {[info exists opts(-m)]} {
	global ${mode}::magicPrefix
	set ${mode}::magicPrefix $opts(-m)
    }
    append lineRegexp "^(\[ \t\]*)"
    lappend lineVars space
    ensureset ${mode}::specialChars ""
    if {[info tclversion] > 8.0} {
	if {[info exists opts(-e)]} {
	    append lineRegexp "(?:(" $opts(-e) ").*)?"
	    lappend lineVars comment
	}
	if {[info exists opts(-s)]} {
	    append lineRegexp "((?!\\\\)\"(?:\\\"|\[^\"\])+(?!\\\\)\")?"
	    lappend lineVars quote
	}
	if {[info exists opts(-i)]} {
	    foreach char $opts(-i) {
		if {[string first $char [set ${mode}::specialChars]] == -1} {
		    append ${mode}::specialChars $char
		}
	    }
	}
    } else {
	if {[info exists opts(-e)]} {
	    append lineRegexp "($opts(-e).*)?"
	    lappend lineVars comment
	}
    }
    
    append lineRegexp "(.*)\$"
    lappend lineVars txt
    if {![info exists ${mode}::lineRegexp]} {
	namespace eval $mode [list set lineRegexp $lineRegexp]
	namespace eval $mode [list set lineVars $lineVars]
    }
    # remember all the old options
    array set ${mode}::keywordsopts [array get opts]
}

proc setRGB {col args} {
    if {[llength $args] != 3} {
	# not sure what this command with no args is supposed to do
	error "Bad args to setRGB"
    }
    set rgb [join $args ""]
    switch -- $col {
	"foreground" {
	    foreach c [info commands .al*.text] {
		$c configure -foreground "#$rgb"
	    }
	}
	"background" {
	    foreach c [info commands .al*.text] {
		$c configure -background "#$rgb"
	    }
	}
	default {
	    foreach c [info commands .al*.text] {
		$c tag configure $col -foreground "#$rgb"
	    }
	}
    }
}

# traceFunc on <funcName> <winName> Trace 'funcName', send output to 
#							'winName'. 
#  traceFunc off			Turn function tracing off.
#  traceFunc status			Display current tracing status.
proc traceFunc {args} {echo "Error: traceFunc called, when it doesn't exist."}
# breakIntoLines <string> - return 'string' with 
#  carriage returns and spaces inserted to satisfy 
#  'leftFillColumn' and 'fillColumn' variables.
proc breakIntoLines {t} {
    global leftFillColumn fillColumn
    set unset {}
    if {![info exists leftFillColumn]} {
	set leftFillColumn 0
	lappend unset leftFillColumn
    }
    if {![info exists fillColumn]} {
	set fillColumn 70
	lappend unset fillColumn
    }
    set width [expr {$fillColumn - $leftFillColumn + 1}]
    if {$t == ""} { return $t }
    regsub -all "\n" $t "\r" t
    #regsub -all "  +" $t " " t
    append t " "
    while 1 {
	if {$t == ""} {
	    break
	}
	set first [string first "\r" $t]
	if {$first != -1 && $first < $width} {
	    append res [string range $t 0 $first]
	    set t [string trimleft [string range $t [expr {$first +1}] end] " "]
	    continue
	}
	set a [string range $t 0 $width]
	set where [string last " " $a]
	if {$where == -1} {
	    set t [string range $t [expr {$width +1}] end]
	} else {
	    set t [string trimleft [string range $t [expr {$where +1}] end] " "]
	    set a [string range $a 0 [expr {$where -1}]]
	}
	append res $a "\r"
    }
    set left [string range "                 " 1 $leftFillColumn]
    regsub -all "(^|\r)" $res "&$left" res
    foreach v $unset {
	unset $v
    }
    return [string trimright $res]
}

# watchCursor - turns the cursor into a a watch cursor.
proc watchCursor {} {
    # not a very good implementation.  May choose wrong window
    # and should really use an idletask not an 'after'
    catch {
	text_cmd configure -cursor watch
	after 500 [list text_cmd configure -cursor xterm]
    }
}

# wc <file>... - counts chars, words, lines of input files.
proc wc {args} {
    foreach f $args {
	wordCount [file::readAll $f]
    }
}

#  save quit undo  #

# quit - quits ALPHA
if {[info commands __quit] == ""} {
    rename quit __quit
}
proc quit {args} {
    # need to check files aren't dirty
    global win::NumDirty
    if {$win::NumDirty > 0} {
	if {![dialog::yesno "There are windows with unsaved changes.  Are you \
	  sure you wish to quit?"]} {
	    return
	}
    }
    
    quitHook
    __quit
}

# save - save current window (or given window)
proc save {{w ""}} {
    global win::Active
    if {$w == ""} {
	set w [lindex $win::Active 0]
    }
    getWinInfo -w $w info
    if {$info(read-only)} {
	return
    }
    set wn [stripNameCount $w]
    if {![file exists $wn]} {
	saveAs [file join [pwd] $wn]
	return
    }
    saveHook $w
    if {[file exists $wn]} {
	if {[catch {file rename -force $wn [file join [file dirname $wn] __tmp_Alpha]}]} {
	    alertnote "Couldn't remove old file.  Save aborted."
	    return
	}
    }
    if {[catch {open $wn w} fout]} {
	alertnote "Sorry, couldn't open the file for writing! Save aborted."
	catch {file rename -force [file join [file dirname $wn] __tmp_Alpha] $wn}
	return
    }
    if {[catch {puts -nonewline $fout [text_wcmd $w get 1.0 "end -1c"]} err]} {
	catch {close $fout}
	catch {file delete -force $wn}
	file rename -force [file join [file dirname $wn] __tmp_Alpha] $wn
	alertnote "Couldn't save; had filesystem error: $err"
    } else {
	close $fout
	catch {file delete -force [file join [file dirname $wn] __tmp_Alpha]}
    }
    ::tw::save $w
    savePostHook $w
}

proc saveAll {} {
    global win::Active
    foreach w $win::Active {
	save $w
    }
}

# saveAs [def name] - save current window with new name. Optionally takes 
#  a default filename. Returns complete path of saved file, if ok hit, 
#  otherwise TCL_ERROR returned.
proc saveAs {{default ""} args} {
    global win::Active win::tk win::tktitle showFullPathsInWindowTitles
    set w [stripNameCount [lindex $win::Active 0]]
    # get new stuff
    if {[llength $args]} {
	if {$default == "-f" && ([llength $args] == 1)} {
	    set name [lindex $args 0]
	} else {
	    error "bad args to saveAs"
	}
    } else {
	if {$default == ""} { set default $w }
	set name [tk_getSaveFile -initialfile $default -filetypes [findFileTypes]]
    }
    if {$name == ""} {
	error "Cancelled"
    }
    if {[file exists $name]} { file delete -force $name }
    set fout [open $name w]
    if {[catch {puts -nonewline $fout [text_cmd get 1.0 "end -1c"]} err]} {
	catch {close $fout}
	catch {file delete -force $name}
	alertnote "Couldn't save; had filesystem error: $err"
    }
    close $fout
    
    #echo "Save as: duplicate window bug in menu possibility"
    set w [lindex $win::Active 0]
    set tkw $win::tk($w)
    unset win::tk($w)
    set win::tk($name) $tkw
    set win::tktitle($tkw) $name
    if {$showFullPathsInWindowTitles} {
	wm title [winfo toplevel $win::tk($name)] $name
    } else {
	wm title [winfo toplevel $win::tk($name)] [file tail $name]
    }
    wm protocol [winfo toplevel $tk($name)] WM_DELETE_WINDOW [list killWindow $name]
    saveasHook [lindex $win::Active 0] $name
    # adjust dirty, undo, redo data.
    ::tw::save $name
    savePostHook $name
}

# redo - redo the next action that has been undone but 
#  not redone
proc redo {} {
    global win::tk win::Current
    tw::redo $win::tk($win::Current)
}
# undo - undo the last action that has not been undone
proc undo {} {
    global win::tk win::Current
    tw::undo $win::tk($win::Current)
}

#  Basic gui stuff  #

# winNames [-f] - return a TCL list of all open windows. If '-f' option 
#  specified, complete pathnames are returned.
proc winNames {{full ""}} {
    global win::Active
    if {$full == "-f"} { return $win::Active }
    return [map "file tail" $win::Active]
}

# sizeWin [win name] <width> <height> - sets size of current or specified window.
#  The window name can be "StatusWin", although only the width can be 
#  changed.
proc sizeWin {w h args} {
    if {[llength $args]} {
	set win $w
	set w $h
	set h [lindex $args 0]
    } else {
	global win::Active
	set win [lindex $win::Active 0]
    }
    global win::tk
    wm geometry [winfo toplevel $win::tk($win)] ${w}x${h}
}
# new [-g <l> <t> <w> <h>] [-n <name>] - opens an untitled window. Can optionally 
#  provide left and top coordinates, plus width and height. All or none.
proc new {args} {
    set i [lsearch -exact $args "-n"]
    if {$i == -1} {
	set i [llength $args]
	lappend args -n "Untitled"
    }
    set n [lindex $args [incr i]]
    set w [alpha::createWin $n]
    set i [lsearch -exact $args "-g"]
    if {$i != -1} {
	foreach {x y w h} [lrange $args [expr {$i +1}] [expr {$i +4}]] {}
	moveWin $n $x $y
	sizeWin $n $w $h
    }
    bringToFront $n
    openHook $n
}
# sendToBack <winName> - Send named window to back.
proc sendToBack {w} {echo "sendToBack $w"}
# setWinInfo [-w <win>] <field> <arg> - Sets a piece of data about either 
#  the current or a specified window. Settable fields 'platform', 'state', 
#  'read-only', 'tabsize', 'dirty', and 'shell'. 'shell' means that dirty 
#  flag ignored and undo off.
proc setWinInfo {field arg args} {
    if {$field == "-w"} {
	set win $arg
	set field [lindex $args 0]
	set arg [lindex $args 1]
    } else {
	set win [win::Current]
    }
    switch -- $field {
	"platform" {
	}
	"state" {
	}
	"readonly" -
	"read-only" {
	    global ::win::tk
	    tw::read_only $win::tk($win) $arg
	}
	"tabsize" {
	    global ::win::tk
	    tw::setTabSize $win::tk($win) $arg
	}
	"dirty" {
	    global ::win::tk
	    tw::dirty $win::tk($win) $arg
	}
	"shell" {
	    global ::tw::shell ::win::tk
	    global ::tw::[set win::tk($win)]
	    if {$arg} {
		set ::tw::[set win::tk($win)](shell) 1
	    } else {
		catch {unset ::tw::[set win::tk($win)](shell)}
	    }
	    
	}  
	default {
	    error "Bad arg '$field' to setWinInfo"
	}
    }
}
# splitWindow [percent] - toggle having window split into two panes. 
#  Optional arg specifies percent of window to allocate to the first pane. 
proc splitWindow {args} {
    text_cmd toggleSplit
}

# toggleScrollbar - toggles horizontal scrollbar on frontmost window. 
#  Will not succeed if scrollbar scrolled.
proc toggleScrollbar {} {
    global win::tk win::Current
    set w [winfo toplevel $win::tk($win::Current)]
    if {[winfo exists $w.hscroll]} {
	destroy $w.hscroll
	$w.text configure -xscrollcommand ""
    } else {
	scrollbar $w.hscroll -command "$w.text xview" -orient horizontal
	grid $w.hscroll -sticky sew -column 0 -row 2
	$w.text configure -xscrollcommand "$w.hscroll set"
    }
}

# bringToFront <winName> - Bring named window to front.
proc bringToFront {n {deactivate 1}} {
    global win::tk win::Active
    if {![info exists win::tk($n)]} {
	# it was just the tail of the name
	foreach nm [array names win::tk] {
	    if {[file tail $nm] == $n} {
		set n $nm
		break
	    }
	}
	if {![info exists win::tk($n)]} {
	    error "Window $n not found!"
	}
    }
    if {[set old [lindex $win::Active 0]] != ""} {
	if {$old != $n} {
	    # if this flag wasn't set, we just killed the last window
	    if {$deactivate} {
		deactivateHook $n
	    }
	}
    }
    set w [winfo toplevel $win::tk($n)]

    # if this flag wasn't set, we killed the last window, so always
    # want to activate this one.
    if {!$deactivate || ($old != $n)} {
	activateHook $n
    }
    
    wm deiconify $w
    raise $w ; focus $w.text

}

# closeAll - close all windows
proc closeAll {} {
    global win::Active
    foreach w $win::Active {
	catch {killWindow $w}
    }
}
# edit [-r] [-m] [-c] [-w] [-g <l> <t> <w> <h>] <name> - Open a file in new 
#  window. '-c' means don't prompt for duplicate win if file already open.
#  '-r' means open the file read-only. '-m' means omit the function titlebar 
#  menu and present only the marks titlebar menu, which is labeled with the 
#  contents of 'markLabel'. The '-g' option allows left and top coords to 
#  be specified, plus width, and height. All or none. '-w' allows you to
#  bypass the "Wrap" dialog for files with long rows.
proc edit {args} {
    global win::tk
    getOpts {{-g 4}}
    set n [file::ensureStandardPath [lindex $args end]]
    if {$n == ""} {return}
    set name $n
    if {[info exists win::tk($n)]} {
	if {[info exists opts(-c)] || (![dialog::yesno "Window already open!  Really open a duplicate?"])} {
	    bringToFront $n
	    return
	}
    }
    if {[lsearch -exact [winNames] [file tail $n]] != -1} {
	set num 2
	# open a duplicate
	while {[lsearch -exact [winNames] "[file tail $n] <$num>"] != -1} { 
	    incr num 
	}
	append name " <$num>"
    }
    
    # doesn't check window exists, or file tail name clash
    set fin [open $n r]
    catch {read $fin} text
    close $fin
    global tcl_platform
    if {$tcl_platform(platform) == "windows"} {
	if {![regexp "\[^\n\]\n\[^\n\]" $text] && [regexp "\n\n" $text]} {
	    regsub -all "\n\n" $text "\n" text
	}
    }
    
    set w [alpha::createWin $name $text]
    if {[info exists opts(-g)]} {
	foreach {x y w h} $opts(-g) {}
	moveWin $name $x $y
	sizeWin $name $w $h
    }
    update idletasks
    bringToFront $name
    openHook $name
    if {[info exists opts(-r)] || ![file writable $n]} {
	winReadOnly
    }
    update
}
# revert - revert the file to its last saved version
proc revert {} {
    set n [win::Current]
    if {[file exists $n]} {
	set fin [open $n r]
	if {[catch {read $fin} text]} {
	    alertnote "Couldn't read the saved file's contents!"
	    close $fin
	    return
	}
	close $fin
	getWinInfo w
	set topl $w(currline)
	setWinInfo read-only 0
	deleteText [minPos] [maxPos]
	insertText $text
	setWinInfo dirty 0
	display [rowColToPos $topl 0]
	if {![file writable $n]} {
	    winReadOnly
	}
	message "File '$n' synchronised with version currently on disk"
    } else {
	error "No such file!"
    }
}


# nextWindow - select next window
proc nextWindow {} {
    global win::Active
    bringToFront [lindex $win::Active 1]
}
# otherPane - If window is split, select the other pane.
proc otherPane {args} {
    text_cmd otherPane
}

# prevWindow - select previous window
proc prevWindow {args} {
    global win::Active
    bringToFront [lindex $win::Active 1]
}

# getGeometry [win] - return a TCL list containing the left 
#  edge of the current window, the top, the width, and height.
proc getGeometry {{w ""}} {
    global win::tk win::Current
    if {$w == ""} {set w $win::Current}
    set g [split [winfo geometry [winfo toplevel $win::tk($w)]] "x+."]
    return [concat [lrange $g 2 3] [lrange $g 0 1]]
}
# getMainDevice - return a list containing the left, top, right, and 
#  bottom of the rectangle defining the main device.
proc getMainDevice {} {
    return [list 0 0 [winfo screenwidth .] [winfo screenheight .]]
}
# getWinInfo [-w <win>] <arr> - Creates an array in current context 
#  containing info about either the current or a specified window. Array 
#  has fields 'state', 'platform', 'read-only', 'tabsize', 'split', 
#  'linesdisp' (num lines that can be seen in the window), 'currline' 
#  (first line displayed), and 'dirty'.
proc getWinInfo {ar args} {
    switch -- [llength $args] {
	0 { 
	    global win::Active win::tk
	    set w [lindex $win::Active 0]
	    uplevel [list upvar \#0 ::tw::[set win::tk($w)] $ar]
	}
	2 {
	    set w [lindex $args 0] 
	    set ar [lindex $args 1]
	    uplevel [list upvar \#0 ::tw::[set win::tk($w)] $ar]
	}
	default {
	    error "wrong args to getWinInfo"
	}
    }
    set tkw $win::tk($w)
    set lines [expr {int([$tkw index end])}]
    set yview [$tkw yview]
    set currline [expr {int($lines * [lindex $yview 0])}]
    set linesdisp [expr {int($lines * ([lindex $yview 1] - [lindex $yview 0]))}]
    uplevel [list set ${ar}(currline) $currline]
    uplevel [list set ${ar}(linesdisp) $linesdisp]
    return ""
}
# icon [-f <winName>] [-c|-o|-t|-q] [-g <h> <v>] - Having to do w/ 
#  iconifying windows. '-c' means close (iconify) window, '-o' open, '-t' 
#  toggle open/close, '-q' returns either a '1' for an iconified window or a 
#  '0' for an uniconified window, and '-g' moves the icon to horizontal 
#  position <h> and vertical position 'v'. Options are executed as they 
#  are parsed, so the '-f' option, if present, should always be first. 
proc icon {args} {
    getOpts {-f}
    if {[info exists opts(-f)]} {
	set w $opts(-f)
    } else {
	set w [win::Current]
    }
    global win::tk
    set w [winfo toplevel $win::tk($w)]
    set state [wm state $w]
    if {[info exists opts(-q)]} {
	if {$state == "normal"} { return 0 } else {
	    return 1
	}
    } elseif {[info exists opts(-c)]} {
	wm iconify $w
    } elseif {[info exists opts(-o)]} {
	wm deiconify $w
    } elseif {[info exists opts(-t)]} {
	if {$state == "normal"} { 
	    wm iconify $w
	} else {
	    wm deiconify $w
	}
    }
}
# killWindow - kill current window
proc killWindow {{wn ""} {destroy_in_progress 0}} {
    global win::tk win::Active win::tktitle
    if {$wn == ""} {
	set wn [lindex $win::Active 0]
    }
    # remove any possible bindings which may trigger
    # side-effects (esp. for destroy)
    if {[info exists win::tk($wn)]} {
	if {[winfo exists $win::tk($wn)]} {
	    bindtags $win::tk($wn) $win::tk($wn)
	    set w [winfo toplevel $win::tk($wn)]
	} else {
	    regexp {^\.[^.]+} $win::tk($wn) w
	}
	if {[winfo exists $w]} {
	    getWinInfo -w $wn winfo
	    if {$winfo(dirty)} {
		switch -- [buttonAlert "That window has unsaved changes.  What shall I do?" "Discard Changes" "Save first" "Cancel"] {
		    "Discard Changes" {
			# do nothing
		    }
		    "Save first" {
			save
		    }
		    "Cancel" {
			error "Cancelled"
		    }
		}
	    }
	    wm withdraw $w
	    # All sorts of nasty recursive loops can arise if we don't remove
	    # these two bindings.  Such loops generally result in wish crashing
	    # (obviously not ideal behaviour, and it ought to catch the infinite
	    # loop, but anyway, we should write nice code too ;-)
	    ::bind $w <Destroy> ""
	    wm protocol $w WM_DELETE_WINDOW ""
	    bindtags $w $w
	    tw::windowCleanup $w.text
	    if {!$destroy_in_progress} {
		destroy $w
	    }
	} else {
	    tw::windowCleanup $w.text
	}
    } else {
	echo "Couldn't cleanup $wn"
    }
    
    
    if {[catch {closeHook $wn} err]} {
	echo "Bad error in closehook; please report bug: $err"
    }
    unset win::tktitle($win::tk($wn))
    unset win::tk($wn)
    global tw::$w.text
    catch {unset tw::$w.text}
    # remove the wrapper proc
    if {[info commands $w.text] != ""} {
	rename $w.text ""
    }
    if {[llength $win::Active]} {
	bringToFront [lindex $win::Active 0] 0
    }
}
# moveWin [win name] <left> <top> - moves current or specified window. 
#  The window name can be "StatusWin".
proc moveWin {x y args} {
    if {[llength $args]} {
	set win $x
	set x $y
	set y [lindex $args 0]
    } else {
	global win::Active
	set win [lindex $win::Active 0]
    }
    global win::tk
    wm geometry [winfo toplevel $win::tk($win)] +${x}+${y}
}

#  Time and timing  #
# now
#  Returns the current time as Macintosh seconds. This is the number of seconds that 
#  have elapsed since Midnight Jan 1, 1904.
proc now {} {clock seconds}
# mtime <time> [long|short|abbrev]
#  Returns a date and time string using the Macintosh International Utilities. The 
#  long/short/abbrev specification corresponds to the date. These are the following 
#  formats:
#	short		3/16/92 9:20:46 PM
#	abbrev	Mon, Mar 16, 1992 9:20:49 PM
#	long		Monday, March 16, 1992 9:20:43 PM
#  The returned value actually is in the form of a list. To get text as 
#  above, run the result through 'join', as in "join [mtime [now] short]".
proc mtime {when {how "short"}} {
    switch -- $how {
	"long" {
	    set c [clock format $when]
	    return [list [concat [lrange $c 0 2] [lindex $c end]] [lindex $c 3]]
	}
	"short" {
	    return [clock format $when -format [list "%m/%d/%Y" "%H:%M:%S %p"]]
	}
	"abbrev" {
	    echo "mtime $when $how not complete"
	    return [clock format $when -format "%m/%d/%Y %H:%M:%S %p"]
	}
    }
}
# ticks
#  Returns the current TickCount. Ticks are 60ths of a seconds. TickCount is the 
#  number of ticks since the Macintosh was started. The command:
#		puts stdout [expr "[ticks] / 60"]
#  will print the number of seconds since the Macintosh was booted.
proc ticks {} {clock clicks}
# iterationCount - allows actions to be repeated many times. "control-u 44 
#  =" inserts 44 '='s into current window.  Also can be used to execute any 
#  function or macro (including the keyboard macro) many times.  Defaults to 
#  4.
proc iterationCount {args} {echo "iterationCount $args"}

#  Printing  #
# pageSetup - display the printing PageSetup dialog.
proc pageSetup {} {
    alertnote "There are currently no 'page setup' options.  Just print away..."
}
# print - print front window
proc print {{f ""}} {
    global printerFont printerFontSize
    if {$f == ""} {
	set f [win::Current]
    }
    global tcl_platform
    switch -- $tcl_platform(platform) {
	"unix" {
	    exec enscript $f
	}
	"windows" {
	    if {$printerFont != ""} {
		print_file $f 0 "$printerFont $printerFontSize"
	    } else {
		print_file $f 0
	    }
	}
	"macintosh" {
	}
    }
    
}

proc printAll {} {
    global win::Active
    foreach f $win::Active {
	print $f
    }
}

#  Filesystem  #

# cp <fromName>+ <toName>
#  This command will copy the file fromName and name the new file toName, 
#  overwriting any existing file. This command copies both data forks, and 
#  the Finder information. 
proc cp {args} {eval file copy $args}
# mkdir <name> - creates a directory (folder) named 
#  'name' in the current directory.
proc mkdir {args} {eval file mkdir $args}
# moveFile <fromName> <toName>
#  This command will move the file fromName to toName, overwriting any 
#  existing file. The move can not be made across volume (disk drives) 
#  boundaries. 
proc moveFile {args} {eval file rename $args}
proc copyFile {args} {eval file copy $args}
# removeFile <fileName>
#  This command will delete the file or folder 'fileName'.
proc removeFile {args} {eval file delete $args}
# rmdir <dirname> - remove a directory (folder)
proc rmdir {args} {eval file delete $args}



#  Not that important  #

# # largestPrefix <list> - Returns the longest prefix contained in all 
#  strings of 'list'.
proc largestPrefix {list} {
    # we only use this where the list is alphabetical
    set first [lindex $list 0]
    set last [lindex $list end]
    set len [string length $first]
    set i 0
    while {[string index $first $i] == [string index $last $i]} {
	if {$i == $len} {
	    break
	}
	incr i
    }
    return [string range $first 0 [expr {$i -1}]]
}

# keyAscii - insert ascii representation (in decimal)
#  of the keydown event, plus a modifier string, if 
#  necessary.
proc keyAscii {args} {echo "keyAscii $args"}
# keyCode - insert the key code along w/ a string 
#  representing and modifiers into the current window.
#  Can be used to create bindings in 'Alphabits'.
proc keyCode {args} {echo "keyCode $args"}

#  Even less important  #

# zapInvisibles - removes chars < ascii 32, except for
#  LF's and CR's.
proc zapInvisibles {args} {echo "zapInvisibles $args"}
# abortEm - aborts whatever is currently happening
proc abortEm {args} {echo "abortEm $args"}
# abbrev <label> <string> [<mode>] - register a label for <string>. See 
#'execAbbrev'.
proc abbrev {args} {echo "abbrev $args"}
# backColor - set background color
proc backColor {args} {echo "backColor $args"}
# dumpColors - dump current foreground and background
#  colors into the current buffer in Alpha-readable
#  format.
proc dumpColors {args} {echo "dumpColors $args"}
# currentPosition - displays current and total bytes.
proc currentPosition {} {
    alertnote "Current: [getPos], maximum: [maxPos]"
}
# execAbbrev - looks at current word and tries to expand it. Labels are 
#  specified using 'abbrev'.
proc execAbbrev {args} {echo "execAbbrev $args"}
# execute - prompt user for a function or macro. The 
#  tab key acts as a "completion" command.
proc execute {args} {
    alertnote "Sorry, 'execute' not yet implemented."
}
# findTag - prompt user for a function name and attempt 
#  to use the file 'cTAGS' to locate the function's 
#  definition
proc findTag {args} {echo "findTag $args"}
# fileInfo - prompts for a file, and displays type, 
#  creator, sizes of both data and resource forks, last
#  modification time, and creation time
proc fileInfo {} {
    set f [getfile]
    foreach {a v} [file attributes $f] {
	append res "[string range $a 1 end] : $v\n"
    }
    alertnote $res
}
# fileRemove - prompts for a file, and removes it
proc fileRemove {} {file delete [getfile "Delete which file?"]}
# freeMem - give a rough approximation of the current 
#  memory reserves of ALPHA
proc freeMem {args} {echo "freeMem $args"}
# getAscii - displays the ASCII code for character at 
#  current insertion point
proc getAscii {args} {echo "getAscii $args"}
# getColors - returns list of colors/hypertext for current document. 
#  Format is list of lists, each sublist consisting of file offset, color 
#  index, and possibly a hypertext command.
proc getColors {args} {echo "getColors $args"}
# insertAscii - prompts for an ASCII code and inserts
#  into text.
proc insertAscii {args} {echo "insertAscii $args"}
# insertFile - prompts for a file name and inserts the
#  corresponding file into the current window. Not
#  undoable.
proc insertFile {args} {echo "insertFile $args"}
# insertPathName - present the user w/ a SFGetFIle dialog 
#  and paste the complete path-name of the chosen file
#  into the current window
proc insertPathName {args} {echo "insertPathName $args"}
# mousePos - Returns list <row,col> of mouse position, if the mouse is 
#  currently over the active window. Otherwise, return error (catch w/ 
#  'catch').
proc mousePos {args} {echo "mousePos $args"}
# thinkReference <-t|-l> <think reference page> - Interact with Think 
#  Reference 2.0. '-t' retrieves a template without leaving Alpha, '-l' 
#  switches to Think Reference and goes to the specified page.
proc thinkReference {args} {echo "thinkReference $args"}

#  Inter-application communcation  #

# AEBuild [<flags>] <app (name|creator)> <aesuite> <aeevent> [<event parameters>]*  -
#  Build and send an apple-event. 'AEBuild' is apple's code of the same 
#  name. Each "event parameter" is two parameters, a parameter type and 
#  the data for the parameter. See the AEBuild doc for more details, 
#  :Tcl:SystemCode:think.tcl for examples. The flags are '-r' (wait for reply), 
#  '-q' (queue reply, in which case 'handleReply' (in appleEvents.tcl) is 
#  called with the reply as a parameter, and '-t <timeout>' specifies the 
#  timeout in ticks.
proc AEBuild {args} {echo "AEBuild $args"}
proc coerce {args} {echo "coerce $args"}

# dosc [<-c 'sign' | -n appName>] [-k 'clas'] [-e 'evnt'] <-s string | -f fileName> [<-t timeout|-r>]
#  Send an AppleEvent, by default class 'misc', event 'dosc'.
#   -c 'sign' 
#   		"sign" is a four-letter creator signature of a running application.
#   -n appName
#   		"appName" is the name of a running application.
#   -k 'clas'
#   		"clas" is a four-letter event class.
#   -e 'evnt'
#   		"evnt" is a four-letter event class.
#   -s string
#   		"string" is the text of a script to send to the other app.
#   -f fileName
#   		"fileName" is the complete or relative pathname of a file the other 
#   		application should execute.
#   -t timeout
#   		"timeout" is the number of ticks Alpha should wait for a response. A 
#   		timeout of "0" means wait forever.
#   -r
#   		Do not wait for reply.
#   -q
#   		Queue reply.
#   		
#   If neither of the '-c' and '-n' options is chosen, the PPC Browser is 
#   used. Either '-s' or '-f' must be chosen. Thus, you can have "Alpha 
#   5.02" and "Alpha 5.02 Copy" sending Apple events to each other, or they 
#   can send events to a remote server such as Apple's ToolServer. 
#   "$HOME:Tcl:UserCode:createStuffitArchive.tcl" contains examples of 
#   controlling Stuffit Deluxe via this command.
proc dosc {args} {echo "dosc $args"}

# launch -f <name> - launch the named app into the background. Note that 
#  for some yet unexplained reason, some applications (MicroSoft Word) 
#  won't launch completely in the background. 'launch'ing such 
#  applications won't insert the application into any system menu that 
#  specifies running applications (although "About the Finder..." will 
#  list it. The only way to get to such an app is through Alpha's 
#  'switchTo', after which the application will finish launching. The '-f' 
#  option gets around this by launching the application in the foreground 
#  instead.
proc launch {args} {
    global tcl_platform
    switch $tcl_platform(platform) {
	"macintosh" {
	    set dosc "tell application \"$app\"\n \"\"\nend tell"
	    AppleScript execute $dosc
	}
	"windows" -
	"unix" {
	    if {[lindex $args 0] == "-f"} {
		exec [lindex $args 1] &
	    } else {
		exec [lindex $args 0]
	    }
	}
    }
}
# switchTo <appName> - Switches to application 'appName'.
proc switchTo {app} {
    global tcl_platform
    switch $tcl_platform(platform) {
	"macintosh" {
	    set dosc "tell application \"$app\"\n \"\"\nend tell"
	    AppleScript execute $dosc
	}
	"windows" -
	"unix" {
	    # nothing
	    echo "switchTo $app ineffective."
	}
    }
}

# processes - returns info of active processes. A list of lists, each 
#  sublist contain a file-name, an application signature, the application 
#  memory size, and the number of ticks the application has been active.
proc processes {args} {echo "processes $args"}

# icURL <URL> - passes arg to Internet Config, if present. Error if not 
#  present. 
proc icURL {args} {
    global tcl_platform
    if {$tcl_platform(platform) == "macintosh"} {
	echo "icURL not implemented yet"
    } else {
	global browserSig
	eval exec [list $browserSig] $args
    }
}

# icGetPref [<-t <type>] <pref name> - Gets preference from 
#  Internet Config. '-t' allows type to be set, '0' returns a string 
#  (default), '1' returns a path name, commonly used for helper apps. A 
#  <pref name> of 'all' returns all valid preferences.
proc icGetPref {args} {echo "icGetPref $args"}
# icOpen - Opens Internet Config
proc icOpen {args} {echo "icOpen $args"}
# nameFromAppl '<app sig>' - Interrogates the desktop database for the first 
#  existing app that has the desired signature. <app sig> is four chars 
#  inside single quotes.
proc nameFromAppl {sig} {
    if {[file exists $sig]} {
	return $sig
    } else {
        # fails if not known (use catch to call this procedure)
	global app::paths
	return $app::paths($sig)
    }
    
}

# sendOpenEvent [filler] <app name> <file name> - Send an open doc event to 
#  another currently running application. If 'filler' is noReply, then a 
#  reply is not requested. Otherwise, we wait for a reply and 'filler' is 
#  ignored. 
proc sendOpenEvent {filler app filename} {
    if {$filler == "noReply"} {
	exec $app $filename &
    } else {
	return [exec $app $filename]
    }
}

# specToPathName [hex data] - given an FSSpec in hex form (as that 
#  returned by aebuild, for instance, see the definition of thinkFileName 
#  in :Tcl:SystemCode:think.tcl), return a complete pathname.
proc specToPathName {args} {echo "specToPathName $args"}
# eventHandler [class] [event] [proc] - Register 'proc' to handle the class 
#  and event specified. Class and event are each 4 char args. 'proc' takes 
#  a single argument that specifies the incoming event according to the 
#  syntax here. There is currently no provision for a reply.
proc eventHandler {args} {echo "eventHandler $args"}



