import ch.bfh.lpdg.DependencyScanner;
import ch.bfh.lpdg.LatexHelper;
import ch.bfh.lpdg.datastructure.DependencyType;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.Collections;

import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;

public class LatexHelperTest {

    private Path copyFile(File source) throws IOException {
        int lastDotIndex = source.getName().lastIndexOf(".");
        var extension = (lastDotIndex != -1) ? source.getName().substring(lastDotIndex) : "";
        return Files.copy(Paths.get(source.getAbsolutePath()), Paths.get(source.getPath().replace(extension, ".temp" + extension)), StandardCopyOption.REPLACE_EXISTING);
    }

    private void removeFile(Path file) throws IOException {
        Files.delete(file);
    }

    private Boolean checkForFile(String filePath) {
        File file = new File(filePath);
        return file.exists() && file.isFile();
    }

    @AfterEach
    public void cleanup() throws IOException {
        removeFile(Paths.get("src/test/resources/LaTexFolder/file1.temp.cls"));
        removeFile(Paths.get("src/test/resources/LaTexFolder/file2.temp.tex"));
        removeFile(Paths.get("src/test/resources/LaTexFolder/file3.temp.sty"));
    }

    @Test
    public void testUnusedPackages() throws IOException {
        var path1 = "src/test/resources/LaTexFolder/file1.cls";
        var copy1 = copyFile(new File(path1));
        var path2 = "src/test/resources/LaTexFolder/file2.tex";
        var copy2 = copyFile(new File(path2));
        var path3 = "src/test/resources/LaTexFolder/file3.sty";
        var copy3 = copyFile(new File(path3));


        var dep1 = DependencyScanner.getInstance().findDependencies(copy1.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);
        var dep2 = DependencyScanner.getInstance().findDependencies(copy2.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);
        var dep3 = DependencyScanner.getInstance().findDependencies(copy3.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);

        var helper1 = new LatexHelper(copy1.toFile(), dep1);
        var helper2 = new LatexHelper(copy2.toFile(), dep2);
        var helper3 = new LatexHelper(copy3.toFile(), dep3);

        helper1.findUnnecessaryDependencies(false, false);
        helper2.findUnnecessaryDependencies(false, false);
        helper3.findUnnecessaryDependencies(false, false);

        var expectedUnusedDependencies = Arrays.asList("xcolor", "makecell");

        for (var dep : dep1.getDependencyList()) {
            if (expectedUnusedDependencies.contains(dep.getName()))
                assertTrue(dep.IsUnused);
            else
                assertFalse(dep.isIsUnused());
        }

        for (var dep : dep2.getDependencyList()) {
            if (expectedUnusedDependencies.contains(dep.getName()))
                assertTrue(dep.IsUnused);
            else
                assertFalse(dep.isIsUnused());
        }

        for (var dep : dep3.getDependencyList()) {
            if (expectedUnusedDependencies.contains(dep.getName()))
                assertTrue(dep.IsUnused);
            else
                assertFalse(dep.isIsUnused());
        }
    }

    @Test
    public void testMinimize() throws IOException {
        var path1 = "src/test/resources/LaTexFolder/file1.cls";
        var copy1 = copyFile(new File(path1));
        var path2 = "src/test/resources/LaTexFolder/file2.tex";
        var copy2 = copyFile(new File(path2));
        var path3 = "src/test/resources/LaTexFolder/file3.sty";
        var copy3 = copyFile(new File(path3));

        var dep1 = DependencyScanner.getInstance().findDependencies(copy1.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);
        var dep2 = DependencyScanner.getInstance().findDependencies(copy2.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);
        var dep3 = DependencyScanner.getInstance().findDependencies(copy3.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);

        var helper1 = new LatexHelper(copy1.toFile(), dep1);
        var helper2 = new LatexHelper(copy2.toFile(), dep2);
        var helper3 = new LatexHelper(copy3.toFile(), dep3);

        helper1.findUnnecessaryDependencies(false, true);
        helper2.findUnnecessaryDependencies(false, true);
        helper3.findUnnecessaryDependencies(false, true);

        assertTrue(checkForFile("src/test/resources/LaTexFolder/file1.temp.minimized.cls"));
        assertTrue(checkForFile("src/test/resources/LaTexFolder/file2.temp.minimized.tex"));
        assertTrue(checkForFile("src/test/resources/LaTexFolder/file3.temp.minimized.sty"));

        removeFile(Paths.get("src/test/resources/LaTexFolder/file1.temp.minimized.cls"));
        removeFile(Paths.get("src/test/resources/LaTexFolder/file2.temp.minimized.tex"));
        removeFile(Paths.get("src/test/resources/LaTexFolder/file3.temp.minimized.sty"));
    }

    @Test
    public void testOverwrite() throws IOException {
        var path1 = "src/test/resources/LaTexFolder/file1.cls";
        var copy1 = copyFile(new File(path1));
        var path2 = "src/test/resources/LaTexFolder/file2.tex";
        var copy2 = copyFile(new File(path2));
        var path3 = "src/test/resources/LaTexFolder/file3.sty";
        var copy3 = copyFile(new File(path3));

        var dep1 = DependencyScanner.getInstance().findDependencies(copy1.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);
        var dep2 = DependencyScanner.getInstance().findDependencies(copy2.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);
        var dep3 = DependencyScanner.getInstance().findDependencies(copy3.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);

        var helper1 = new LatexHelper(copy1.toFile(), dep1);
        var helper2 = new LatexHelper(copy2.toFile(), dep2);
        var helper3 = new LatexHelper(copy3.toFile(), dep3);

        helper1.findUnnecessaryDependencies(true, false);
        helper2.findUnnecessaryDependencies(true, false);
        helper3.findUnnecessaryDependencies(true, false);

        assertFalse(checkForFile("src/test/resources/LaTexFolder/file1.temp.minimized.cls"));
        assertFalse(checkForFile("src/test/resources/LaTexFolder/file2.temp.minimized.tex"));
        assertFalse(checkForFile("src/test/resources/LaTexFolder/file3.temp.minimized.sty"));
        assertTrue(checkForFile("src/test/resources/LaTexFolder/file1.temp.cls"));
        assertTrue(checkForFile("src/test/resources/LaTexFolder/file2.temp.tex"));
        assertTrue(checkForFile("src/test/resources/LaTexFolder/file3.temp.sty"));
    }

    @Test
    public void testMinimizeAndOverwrite() throws IOException {
        var path1 = "src/test/resources/LaTexFolder/file1.cls";
        var copy1 = copyFile(new File(path1));
        var path2 = "src/test/resources/LaTexFolder/file2.tex";
        var copy2 = copyFile(new File(path2));
        var path3 = "src/test/resources/LaTexFolder/file3.sty";
        var copy3 = copyFile(new File(path3));

        var dep1 = DependencyScanner.getInstance().findDependencies(copy1.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);
        var dep2 = DependencyScanner.getInstance().findDependencies(copy2.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);
        var dep3 = DependencyScanner.getInstance().findDependencies(copy3.toAbsolutePath().toString(), DependencyType.FILE, "", Collections.emptyList(), 5);

        var helper1 = new LatexHelper(copy1.toFile(), dep1);
        var helper2 = new LatexHelper(copy2.toFile(), dep2);
        var helper3 = new LatexHelper(copy3.toFile(), dep3);

        helper1.findUnnecessaryDependencies(true, true);
        helper2.findUnnecessaryDependencies(true, true);
        helper3.findUnnecessaryDependencies(true, true);

        assertFalse(checkForFile("src/test/resources/LaTexFolder/file1.temp.minimized.cls"));
        assertFalse(checkForFile("src/test/resources/LaTexFolder/file2.temp.minimized.tex"));
        assertFalse(checkForFile("src/test/resources/LaTexFolder/file3.temp.minimized.sty"));
        assertTrue(checkForFile("src/test/resources/LaTexFolder/file1.temp.cls"));
        assertTrue(checkForFile("src/test/resources/LaTexFolder/file2.temp.tex"));
        assertTrue(checkForFile("src/test/resources/LaTexFolder/file3.temp.sty"));
    }
}
